function grid_img = display_grid(varargin)
%this function allow you to create an image with a grid pattern of any
%size, and it will allow you to draw on that grid squares and other objects(arrows)
%via two corner specification.
%syntax: grid_img = display_grid('size',[640 480],'grid',[80 80],...
%           'square',[80 80 160 160],...
%           'background',0,'grid_color',255,'draw_color',255);
%Input: 'size' = the size of the image. Example: 640x480.  Default: screen
%           size.
%       'grid' = the grid size. Example: 80x80 grids across 480x640 image.
%           Note: 'grid' by default is on, and is 10x10, but the 'off'
%           command can turn off the grid.
%       'square' = the location of the square by defining the upper left
%       and the lower right corner.  An array can be used to display more
%       than one square.  Thus 5 squares will be a 5 row 4 col array.
%       'background' = background color.  Default = 0 (black);
%       'grid_color' = color of the grid lines.  Default = 255 (white,
%       uint8);
%       'draw_color' = Color of square, etc.  Default = 255 (white)
%       'line_size' = the line size of the grid in pixels.  Default = 1
%Output: grid_img = the output image.

[img_size,grid,grid_status,background,grid_color,line_size,draw_img] = parse(varargin);

%initate image. note: pixel coordinates and array coordinates in matlab is
%flipped
grid_img = ones(img_size(1,2),img_size(1,1),'uint8')*background;

%if grid is on draw the grid
if grid_status
    %draw vertical lines first
    for i = 0:grid(1,1):img_size(1,1)
        if i==0     %0 is a special case
            grid_img(:,1:line_size) = grid_color;
        elseif i==img_size(1,1)    %last line is special too if it is at the edge
            grid_img(:,i-line_size:i) = grid_color;
            %note there is actually the special case when the line width is
            %larger than the remaining pixels of i to the edge, but I am
            %too lazy for now to fix this case that will probably never
            %come up.
        else
            grid_img(:,i:i+line_size-1) = grid_color;
        end
    end

    %draw horizontal lines
    for i = 0:grid(1,2):img_size(1,2)
        if i==0     %0 is a special case
            grid_img(1:line_size,:) = grid_color;
        elseif i==img_size(1,2)    %last line is special too if it is at the edge
            grid_img(i-line_size:i,:) = grid_color;
        else
            grid_img(i:i+line_size-1,:) = grid_color;
        end
    end
end

%draw images if there are ones specified
if ~isempty(draw_img(1).type)
    %draw the images
    for i = 1:size(draw_img(1).type,1)
        switch draw_img(1).type{i,1}
            case 'square'
                sqr_idx = draw_img(1).square;   %pull out the index for ease of use.
                %use logical indexing to make sure there are no 0 indexing
                %in our designation arrays
                sqr_idx(sqr_idx==0) = 1;      %0 = 1 instead
                %draw all the squares
                for j = 1:size(sqr_idx,1)
                    grid_img(sqr_idx(j,2):sqr_idx(j,4),sqr_idx(j,1):sqr_idx(j,3)) = draw_img(1).color;
                end
        end
    end
end
          

%--------------------------------------------------------------------------
%Subfunction to parse the input
function [img_size,grid,grid_status,background,grid_color,line_size,draw_img] = parse(input)
%Initiate Parameter Space
img_size = get(0,'screensize');
img_size = img_size(3:4);   %default = screen size
grid_status = 1;    %on
grid = [10 10];     %default = 10x10 grid
background = 0;     %default = black (uint8)
grid_color = 255;   %default = white (uint8)
line_size = 1;      %default in pixels
draw_img.type = [];      %draw nothing.
draw_img.color = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2);
        if ischar(input{1,i})
            switch input{1,i}
                case 'size'
                    if isnumeric(input{1,i+1})
                        img_size = input{1,i+1};
                    else
                        error(['Error: parameters for size must follow the form: [x y]']);
                    end
                case 'grid'
                    if isnumeric(input{1,i+1})
                        grid = input{1,i+1};
                    elseif strcmp('off',input{1,i+1})
                        grid_status = 0;
                    else
                        error(['Error: parameters for grid must follow the form: [x y] or ','off''']);
                    end
                case 'background'
                    if isnumeric(input{1,i+1})
                        background = input{1,i+1};
                    else
                        error(['Error: parameters for background must follow the form: x where x is an integer between 0 and 255']);
                    end
                case 'grid_color'
                    if isnumeric(input{1,i+1})
                        grid_color = input{1,i+1};
                    else
                        error(['Error: parameters for grid_color must follow the form: x where x is an integer between 0 and 255']);
                    end
                case 'line_size'
                    if isnumeric(input{1,i+1})
                        line_size = input{1,i+1};
                    else
                        error(['Error: parameters for line_size must follow the form: [x]']);
                    end
                case 'draw_color'
                    if isnumeric(input{1,i+1})
                        draw_img.color = input{1,i+1};
                    else
                        error(['Error: parameters for draw_color must follow the form: x where x is an integer between 0 and 255']);
                    end
                case 'square'
                    if isnumeric(input{1,i+1})
                        if isempty(draw_img.type)
                            draw_img.type{1,1} = 'square';
                        else
                            a = size(draw_img(1).type,1)+1
                            draw_img.type{a,1} = 'square';
                        end
                        if isempty(draw_img.color)
                            draw_img.color = 255;   %default
                        end
                        draw_img(1).square = input{1,i+1};
                    else
                        error(['Error: parameters for square must follow the form: [x1 x2 y1 y2;x3 x4 y3 y4]']);
                    end
                otherwise
                    error(['Error: ',input{1,i},' is not a valid input to display_grid.']);
            end
        else
            error(['Error: You have entered an improper argument. Please check Help for how to use display_grid.']);
        end
    end
end